// AS5149.1 Refrigerant Charge Calculator - Complete Implementation

// Occupancy Category Lookup Table (Tables!A4:B50)
const occupancyCategoryTable = {
  "House": "a",
  "Residential": "a",
  "Hospital": "a",
  "Court": "a",
  "Prison": "a",
  "Theatre": "a",
  "Supermarket (Public Areas)": "a",
  "School": "a",
  "Lecture hell": "a",
  "Library": "a",
  "Public Transport Terminal": "a",
  "Hotel": "a",
  "Dwelling": "a",
  "Mall": "a",
  "Restaurant": "a",
  "Any Public Area": "a",
  "Business office": "b",
  "Laboratory": "b",
  "General Manufacturing": "b",
  "Any General Work Place": "b",
  "Special Manufacturing Facility": "c",
  "Cold Stores": "c",
  "Dairy": "c",
  "Abattoiry": "c",
  "Supermarket (Non-Public Areas)": "c",
  "Machinery Room": "c"
};

// Location Class Lookup Table (Tables!D4:E7)
const locationClassTable = {
  "Mechanical Equipment within the occupied Space": "I",
  "Compressor within the machinery room or open air": "II",
  "Machinery room or open Air": "III",
  "Ventilated enclosures": "IV"
};

// Refrigerant Parameters Table (Tables!G4:L146)
const refrigerantParamsTable = {
  "R11": { toxicity: "A", flammability: 1, practical_l: 0.3, atel_odl: 0.0062, lfl: "NF" },
  "R12": { toxicity: "A", flammability: 1, practical_l: 0.5, atel_odl: 0.088, lfl: "NF" },
  "R12B1": { toxicity: "N", flammability: "D", practical_l: 0.2, atel_odl: "ND", lfl: "NF" },
  "R13": { toxicity: "A", flammability: 1, practical_l: 0.5, atel_odl: "ND", lfl: "NF" },
  "R13B1": { toxicity: "A", flammability: 1, practical_l: 0.6, atel_odl: "ND", lfl: "NF" },
  "R14": { toxicity: "A", flammability: 1, practical_l: 0.4, atel_odl: 0.4, lfl: "NF" },
  "R22": { toxicity: "A", flammability: 1, practical_l: 0.3, atel_odl: 0.21, lfl: "NF" },
  "R23": { toxicity: "A", flammability: 1, practical_l: 0.68, atel_odl: 0.15, lfl: "NF" },
  "R30": { toxicity: "B", flammability: 1, practical_l: 0.017, atel_odl: "ND", lfl: "NF" },
  "R32": { toxicity: "A", flammability: "2L", practical_l: 0.061, atel_odl: 0.3, lfl: 0.307 },
  "R50": { toxicity: "A", flammability: 3, practical_l: 0.006, atel_odl: "ND", lfl: 0.032 },
  "R113": { toxicity: "A", flammability: 1, practical_l: 0.4, atel_odl: 0.02, lfl: "NF" },
  "R114": { toxicity: "A", flammability: 1, practical_l: 0.7, atel_odl: 0.14, lfl: "NF" },
  "R115": { toxicity: "A", flammability: 1, practical_l: 0.76, atel_odl: 0.76, lfl: "NF" },
  "R116": { toxicity: "A", flammability: 1, practical_l: 0.68, atel_odl: 0.68, lfl: "NF" },
  "R123": { toxicity: "B", flammability: 1, practical_l: 0.1, atel_odl: 0.057, lfl: "NF" },
  "R124": { toxicity: "A", flammability: 1, practical_l: 0.11, atel_odl: 0.056, lfl: "NF" },
  "R125": { toxicity: "A", flammability: 1, practical_l: 0.39, atel_odl: 0.37, lfl: "NF" },
  "R134a": { toxicity: "A", flammability: 1, practical_l: 0.25, atel_odl: 0.21, lfl: "NF" },
  "R141b": { toxicity: "ND", flammability: "ND", practical_l: 0.053, atel_odl: 0.012, lfl: 0.363 },
  "R142b": { toxicity: "A", flammability: 2, practical_l: 0.049, atel_odl: 0.1, lfl: 0.329 },
  "R143a": { toxicity: "A", flammability: "2L", practical_l: 0.048, atel_odl: 0.48, lfl: 0.282 },
  "R152a": { toxicity: "A", flammability: 2, practical_l: 0.027, atel_odl: 0.14, lfl: 0.13 },
  "R170": { toxicity: "A", flammability: 3, practical_l: 0.0086, atel_odl: 0.0086, lfl: 0.038 },
  "R1150": { toxicity: "A", flammability: 3, practical_l: 0.006, atel_odl: "ND", lfl: 0.036 },
  "RE170": { toxicity: "A", flammability: 3, practical_l: 0.013, atel_odl: 0.079, lfl: 0.064 },
  "R218": { toxicity: "A", flammability: 1, practical_l: 1.84, atel_odl: 0.85, lfl: "NF" },
  "R227ea": { toxicity: "A", flammability: 1, practical_l: 0.63, atel_odl: 0.63, lfl: "NF" },
  "R236fa": { toxicity: "A", flammability: 1, practical_l: 0.59, atel_odl: 0.34, lfl: "NF" },
  "R245fa": { toxicity: "B", flammability: 1, practical_l: 0.19, atel_odl: 0.19, lfl: "NF" },
  "R290": { toxicity: "A", flammability: 3, practical_l: 0.008, atel_odl: 0.09, lfl: 0.038 },
  "R1234yf": { toxicity: "A", flammability: "2L", practical_l: 0.058, atel_odl: 0.47, lfl: 0.289 },
  "R1234ze(E)": { toxicity: "A", flammability: "2L", practical_l: 0.061, atel_odl: 0.28, lfl: 0.303 },
  "R1270": { toxicity: "A", flammability: 3, practical_l: 0.008, atel_odl: 0.0017, lfl: 0.046 },
  "R600": { toxicity: "A", flammability: 3, practical_l: 0.0089, atel_odl: 0.0024, lfl: 0.038 },
  "R600a": { toxicity: "A", flammability: 3, practical_l: 0.011, atel_odl: 0.059, lfl: 0.043 },
  "R601": { toxicity: "A", flammability: 3, practical_l: 0.008, atel_odl: 0.0029, lfl: 0.035 },
  "R601a": { toxicity: "A", flammability: 3, practical_l: 0.008, atel_odl: 0.0029, lfl: 0.038 },
  "RC318": { toxicity: "A", flammability: 1, practical_l: 0.81, atel_odl: 0.65, lfl: "NF" },
  "R717": { toxicity: "B", flammability: "2L", practical_l: 0.00035, atel_odl: 0.00022, lfl: 0.116 },
  "R744": { toxicity: "A", flammability: 1, practical_l: 0.1, atel_odl: 0.072, lfl: "NF" },
  "R401A": { toxicity: "A", flammability: 1, practical_l: 0.3, atel_odl: 0.1, lfl: "NF" },
  "R401B": { toxicity: "A", flammability: 1, practical_l: 0.34, atel_odl: 0.11, lfl: "NF" },
  "R401C": { toxicity: "A", flammability: 1, practical_l: 0.24, atel_odl: 0.083, lfl: "NF" },
  "R402A": { toxicity: "A", flammability: 1, practical_l: 0.33, atel_odl: 0.27, lfl: "NF" },
  "R402B": { toxicity: "A", flammability: 1, practical_l: 0.32, atel_odl: 0.24, lfl: "NF" },
  "R403A": { toxicity: "A", flammability: 2, practical_l: 0.33, atel_odl: 0.24, lfl: 0.48 },
  "R403B": { toxicity: "A", flammability: 1, practical_l: 0.41, atel_odl: 0.29, lfl: "NF" },
  "R404A": { toxicity: "A", flammability: 1, practical_l: 0.52, atel_odl: 0.52, lfl: "NF" },
  "R405A": { toxicity: "ND", flammability: "ND", practical_l: "ND", atel_odl: 0.26, lfl: "ND" },
  "R406A": { toxicity: "A", flammability: 2, practical_l: 0.13, atel_odl: 0.14, lfl: 0.302 },
  "R407A": { toxicity: "A", flammability: 1, practical_l: 0.33, atel_odl: 0.31, lfl: "NF" },
  "R407B": { toxicity: "A", flammability: 1, practical_l: 0.35, atel_odl: 0.33, lfl: "NF" },
  "R407C": { toxicity: "A", flammability: 1, practical_l: 0.31, atel_odl: 0.29, lfl: "NF" },
  "R407D": { toxicity: "A", flammability: 1, practical_l: 0.41, atel_odl: 0.25, lfl: "NF" },
  "R407E": { toxicity: "A", flammability: 1, practical_l: 0.4, atel_odl: 0.27, lfl: "NF" },
  "R407F": { toxicity: "A", flammability: 1, practical_l: 0.32, atel_odl: 0.32, lfl: "NF" },
  "R408A": { toxicity: "A", flammability: 1, practical_l: 0.41, atel_odl: 0.33, lfl: "NF" },
  "R409A": { toxicity: "A", flammability: 1, practical_l: 0.16, atel_odl: 0.12, lfl: "NF" },
  "R409B": { toxicity: "A", flammability: 1, practical_l: 0.17, atel_odl: 0.12, lfl: "NF" },
  "R410A": { toxicity: "A", flammability: 1, practical_l: 0.44, atel_odl: 0.42, lfl: "NF" },
  "R410B": { toxicity: "A", flammability: 1, practical_l: 0.43, atel_odl: 0.43, lfl: "NF" },
  "R411A": { toxicity: "A", flammability: 2, practical_l: 0.04, atel_odl: 0.074, lfl: 0.186 },
  "R411B": { toxicity: "A", flammability: 2, practical_l: 0.05, atel_odl: 0.044, lfl: 0.239 },
  "R412A": { toxicity: "A", flammability: 2, practical_l: 0.07, atel_odl: 0.17, lfl: 0.329 },
  "R413A": { toxicity: "A", flammability: 2, practical_l: 0.08, atel_odl: 0.21, lfl: 0.375 },
  "R414A": { toxicity: "A", flammability: 1, practical_l: 0.1, atel_odl: 0.1, lfl: "NF" },
  "R414B": { toxicity: "A", flammability: 1, practical_l: 0.096, atel_odl: 0.096, lfl: "NF" },
  "R415A": { toxicity: "A", flammability: 2, practical_l: 0.04, atel_odl: 0.19, lfl: 0.188 },
  "R415B": { toxicity: "A", flammability: 2, practical_l: 0.03, atel_odl: 0.15, lfl: 0.13 },
  "R416A": { toxicity: "A", flammability: 1, practical_l: 0.064, atel_odl: 0.064, lfl: "NF" },
  "R417A": { toxicity: "A", flammability: 1, practical_l: 0.15, atel_odl: 0.057, lfl: "NF" },
  "R417B": { toxicity: "A", flammability: 1, practical_l: 0.069, atel_odl: 0.069, lfl: "NF" },
  "R418A": { toxicity: "A", flammability: 2, practical_l: 0.06, atel_odl: 0.2, lfl: 0.31 },
  "R419A": { toxicity: "A", flammability: 2, practical_l: 0.05, atel_odl: 0.31, lfl: 0.25 },
  "R420A": { toxicity: "A", flammability: 1, practical_l: 0.18, atel_odl: 0.18, lfl: "NF" },
  "R421A": { toxicity: "A", flammability: 1, practical_l: 0.28, atel_odl: 0.28, lfl: "NF" },
  "R421B": { toxicity: "A", flammability: 1, practical_l: 0.33, atel_odl: 0.33, lfl: "NF" },
  "R422A": { toxicity: "A", flammability: 1, practical_l: 0.29, atel_odl: 0.29, lfl: "NF" },
  "R422B": { toxicity: "A", flammability: 1, practical_l: 0.25, atel_odl: 0.25, lfl: "NF" },
  "R422C": { toxicity: "A", flammability: 1, practical_l: 0.29, atel_odl: 0.29, lfl: "NF" },
  "R422D": { toxicity: "A", flammability: 1, practical_l: 0.26, atel_odl: 0.26, lfl: "NF" },
  "R423A": { toxicity: "A", flammability: 1, practical_l: 0.3, atel_odl: 0.3, lfl: "NF" },
  "R424A": { toxicity: "A", flammability: 1, practical_l: 0.1, atel_odl: 0.1, lfl: "NF" },
  "R425A": { toxicity: "A", flammability: 1, practical_l: 0.27, atel_odl: 0.27, lfl: "NF" },
  "R426A": { toxicity: "A", flammability: 1, practical_l: 0.083, atel_odl: 0.083, lfl: "NF" },
  "R427A": { toxicity: "A", flammability: 1, practical_l: 0.29, atel_odl: 0.29, lfl: "NF" },
  "R428A": { toxicity: "A", flammability: 1, practical_l: 0.37, atel_odl: 0.37, lfl: "NF" },
  "R429A": { toxicity: "A", flammability: 3, practical_l: 0.01, atel_odl: 0.098, lfl: 0.052 },
  "R430A": { toxicity: "A", flammability: 3, practical_l: 0.017, atel_odl: 0.1, lfl: 0.084 },
  "R431A": { toxicity: "A", flammability: 3, practical_l: 0.009, atel_odl: 0.1, lfl: 0.044 },
  "R432A": { toxicity: "A", flammability: 3, practical_l: 0.008, atel_odl: 0.0021, lfl: 0.039 },
  "R433A": { toxicity: "A", flammability: 3, practical_l: 0.007, atel_odl: 0.0055, lfl: 0.036 },
  "R433B": { toxicity: "A", flammability: 3, practical_l: 0.005, atel_odl: 0.025, lfl: 0.025 },
  "R433C": { toxicity: "A", flammability: 3, practical_l: 0.006, atel_odl: 0.0066, lfl: 0.032 },
  "R434A": { toxicity: "A", flammability: 1, practical_l: 0.32, atel_odl: 0.32, lfl: "NF" },
  "R435A": { toxicity: "A", flammability: 3, practical_l: 0.014, atel_odl: 0.09, lfl: 0.069 },
  "R436A": { toxicity: "A", flammability: 3, practical_l: 0.006, atel_odl: 0.073, lfl: 0.032 },
  "R436B": { toxicity: "A", flammability: 3, practical_l: 0.007, atel_odl: 0.071, lfl: 0.033 },
  "R437A": { toxicity: "A", flammability: 1, practical_l: 0.081, atel_odl: 0.081, lfl: "NF" },
  "R438A": { toxicity: "A", flammability: 1, practical_l: 0.079, atel_odl: 0.079, lfl: "NF" },
  "R439A": { toxicity: "A", flammability: 2, practical_l: 0.061, atel_odl: 0.34, lfl: 0.304 },
  "R440A": { toxicity: "A", flammability: 2, practical_l: 0.025, atel_odl: 0.14, lfl: 0.124 },
  "R441A": { toxicity: "A", flammability: 3, practical_l: 0.0063, atel_odl: 0.0063, lfl: 0.032 },
  "R442A": { toxicity: "A", flammability: 1, practical_l: 0.33, atel_odl: 0.33, lfl: "NF" },
  "R500": { toxicity: "A", flammability: 1, practical_l: 0.4, atel_odl: 0.12, lfl: "NF" },
  "R501": { toxicity: "A", flammability: 1, practical_l: 0.38, atel_odl: 0.21, lfl: "NF" },
  "R502": { toxicity: "A", flammability: 1, practical_l: 0.45, atel_odl: 0.33, lfl: "NF" },
  "R503": { toxicity: "A", flammability: 1, practical_l: 0.35, atel_odl: "ND", lfl: "NF" },
  "R504": { toxicity: "A", flammability: 1, practical_l: 0.45, atel_odl: 0.45, lfl: "NF" },
  "R507A": { toxicity: "A", flammability: 1, practical_l: 0.53, atel_odl: 0.53, lfl: "NF" },
  "R508A": { toxicity: "A", flammability: 1, practical_l: 0.23, atel_odl: 0.23, lfl: "NF" },
  "R508B": { toxicity: "A", flammability: 1, practical_l: 0.25, atel_odl: 0.2, lfl: "NF" },
  "R509A": { toxicity: "A", flammability: 1, practical_l: 0.56, atel_odl: 0.38, lfl: "NF" },
  "R510A": { toxicity: "A", flammability: 3, practical_l: 0.011, atel_odl: 0.087, lfl: 0.056 },
  "R511A": { toxicity: "A", flammability: 3, practical_l: 0.008, atel_odl: 0.092, lfl: 0.038 },
  "R512A": { toxicity: "A", flammability: 2, practical_l: 0.025, atel_odl: 0.14, lfl: 0.124 }
};

// Table A.1 - Toxicity Limit Lookup (Table A.1 sheet)
const tableA1 = [
  { toxicity: "A", category: "a", occupancy_desc: null, class_i: "Toxicity limit x Location volume", class_ii: "Toxicity limit x Location volume", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: "A" },
  { toxicity: "A", category: "b", occupancy_desc: "Below Ground", class_i: "Toxicity limit x Location volume", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: "a" },
  { toxicity: "A", category: "b", occupancy_desc: "Above Ground", class_i: "No charge restrictiona", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: "Above Ground" },
  { toxicity: "A", category: "c", occupancy_desc: "Below Ground", class_i: "Toxicity limit x Location volume", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: "I" },
  { toxicity: "A", category: "c", occupancy_desc: "Above Ground", class_i: "No charge restriction", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: "Toxicity limit x Location volume" },
  { toxicity: "B", category: "a", occupancy_desc: null, class_i: "Toxicity Limit x Location Volume and not more than 2.5 kg for Sealed absorption systems", class_ii: "Toxicity Limit x Location Volume and not more than 2.5 kg for Sealed absorption systems", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: null },
  { toxicity: "B", category: "b", occupancy_desc: "Below Ground", class_i: "Toxicity limit x Location volume", class_ii: "Not more than 25 kg", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: null },
  { toxicity: "B", category: "b", occupancy_desc: "<1PPL/10sqm", class_i: "Not more than 10 kg", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: null },
  { toxicity: "B", category: "b", occupancy_desc: "Above Ground", class_i: "Not more than 10 kg", class_ii: "Not more than 25 kg", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: null },
  { toxicity: "B", category: "c", occupancy_desc: "<1PPL/10sqm", class_i: "Not more than 50 kg and Emer. Exits available", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: null },
  { toxicity: "B", category: "c", occupancy_desc: "Above Ground", class_i: "Not more than 10 kg", class_ii: "Not more than 25 kg", class_iii: "No charge restriction", class_iv: "Change Location Class based on the location of the ventilated enclosure.", selector: null }
];

// Table A.2 - Flammability Limit Lookup (Table A.2 sheet)
const tableA2 = [
  { flammability: "2L", category: "a", application: "Human Comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2 x 1,5", class_ii: "According to A.4 and not more than m2 x 1,5", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: "a" },
  { flammability: "2L", category: "a", application: "Other applications", occupancy_desc: null, class_i: "20 % x LFL x Room volume and not more than m2 x 1,5", class_ii: "20 % x LFL x Room volume and not more than m2 x 1,5 ", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: "Human Comfort" },
  { flammability: "2L", category: "b", application: "Human Comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2 x 1,5", class_ii: "According to A.4 and not more than m2 x 1,5", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: "Above Ground" },
  { flammability: "2L", category: "b", application: "Other applications", occupancy_desc: null, class_i: "20 % x LFL x Room volume and not more than m2 x 1,5", class_ii: "20 % x LFL x Room volume and not more than 25 kg", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: "I" },
  { flammability: "2L", category: "c", application: "Human Comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2 x 1,5", class_ii: "According to A.4 and not more than m2 x 1,5", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: "According to A.4 and not more than m2 x 1,5" },
  { flammability: "2L", category: "c", application: "Other applications", occupancy_desc: null, class_i: "20 % x LFL x Room volume and not more than m2 x 1,5", class_ii: "20 % x LFL x Room volume and not more than 25 kg", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: null },
  { flammability: "2L", category: "c", application: "Other applications", occupancy_desc: "<1PPL/10sqm", class_i: "20 % x LFL x Room volume and not more than 50 kg", class_ii: "No charge restriction", class_iii: "No charge restriction", class_iv: "not more than m3 x 1,5", selector: null },
  { flammability: "2", category: "a", application: "Human Comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2", class_ii: "According to A.4 and not more than m2", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 2, category: "a", application: "Other applications", occupancy_desc: null, class_i: "20 % x LFL x Room volume and not more than m2", class_ii: "20 % x LFL x Room volume and not more than m2", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 2, category: "b", application: "Human Comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2", class_ii: "According to A.4 and not more than m2", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 2, category: "b", application: "Other applications", occupancy_desc: null, class_i: "20 % x LFL x Room volume and not more than m2", class_ii: "20 % x LFL x Room volume and not more than m2", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 2, category: "c", application: "Human Comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2", class_ii: "According to A.4 and not more than m2", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 2, category: "c", application: "Other applications", occupancy_desc: "Below Ground", class_i: "20 % x LFL x Room volume and not more than m2", class_ii: "20 % x LFL x Room volume and not more than m2", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 2, category: "c", application: "Other applications", occupancy_desc: "Above Ground", class_i: "20 % x LFL x Room volume and not more than 10 kg", class_ii: "20 % x LFL x Room volume and not more than 25 kg", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "a", application: "Human comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2", class_ii: "According to A.4 and not more than m2", class_iii: "In accordance with occupancy category a, other applications", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "a", application: "Other applications", occupancy_desc: "Below Ground", class_i: "Only sealed systems: 20% x LFL x Room volume and not more than 1 kg", class_ii: "Only sealed systems: 20% x LFL x Room volume and not more than 1 kg", class_iii: "Not more than 1 kg", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "a", application: "Other applications", occupancy_desc: "Above Ground", class_i: "Only sealed systems: 20% x LFL x Room volume and not more than 1,5 kg", class_ii: "Only sealed systems: 20% x LFL x Room volume and not more than 1,5 kg", class_iii: "Not more than 5 kg", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "b", application: "Human comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2", class_ii: "According to A.4 and not more than m2", class_iii: "In accordance with occupancy category b, other applications", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "b", application: "Other applications", occupancy_desc: "Below Ground", class_i: "20% x LFL x Room volume and not more than 1 kg", class_ii: "20% x LFL x Room volume and not more than 1 kg", class_iii: "Not more than 1 kg", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "b", application: "Other applications", occupancy_desc: "Above Ground", class_i: "20% x LFLx Room volume and not more than 2,5 kg", class_ii: "20% x LFLx Room volume and not more than 2,5 kg", class_iii: "Not more than 10 kg", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "c", application: "Human comfort", occupancy_desc: null, class_i: "According to A.4 and not more than m2", class_ii: "According to A.4 and not more than m2", class_iii: "In accordance with occupancy category c, other applications", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "c", application: "Other applications", occupancy_desc: "Below Ground", class_i: "20% LFLx Room volume and not more than 1 kg", class_ii: "20% LFLx Room volume and not more than 1 kg", class_iii: "Not more than 1 kg", class_iv: "not more than m3", selector: null },
  { flammability: 3, category: "c", application: "Other applications", occupancy_desc: "Above Ground", class_i: "20% x LFL x Room volume and not more than 10 kg", class_ii: "20% x LFL x Room volume and not more than 25 kg", class_iii: "No charge restriction", class_iv: "not more than m3", selector: null }
];

// Helper function for safe value parsing
function parseValue(val) {
  if (val === null || val === undefined || val === "" || val === "ND" || val === "NF") {
    return null;
  }
  if (typeof val === "number") {
    return val;
  }
  if (typeof val === "string") {
    const num = parseFloat(val);
    return isNaN(num) ? null : num;
  }
  return null;
}

// Helper function to check if value is numeric
function isNumeric(val) {
  return val !== null && val !== undefined && val !== "" && val !== "ND" && val !== "NF" && !isNaN(parseFloat(val));
}

// Lookup function for Table A.1 (Toxicity)
function lookupTableA1(toxicity, category, occupancyDesc, locationClass) {
  if (!toxicity || !category || !locationClass) {
    return "No matching rule";
  }
  
  // Normalize user occupancy description
  const userOcc = occupancyDesc === "" || occupancyDesc === null ? null : occupancyDesc;
  
  // Two-pass lookup: First try exact matches (specific occupancy_desc), then general matches (null occupancy_desc)
  // This ensures we prioritize specific rules over general ones
  let exactMatch = null;
  let generalMatch = null;
  
  for (const row of tableA1) {
    const matchTox = row.toxicity === toxicity;
    const matchCat = row.category === category;
    
    if (!matchTox || !matchCat) continue;
    
    // Check for exact occupancy match (both have values and they match)
    if (row.occupancy_desc !== null && userOcc !== null && row.occupancy_desc === userOcc) {
      if (!exactMatch) exactMatch = row;
    }
    // Check for general match (row has null occupancy_desc, meaning it applies to any)
    // This matches when: (1) row has null and user has null, OR (2) row has null and user has a value (general rule)
    else if (row.occupancy_desc === null) {
      if (!generalMatch) generalMatch = row;
    }
  }
  
  // Use exact match if found, otherwise use general match
  const matchedRow = exactMatch || generalMatch;
  
  if (matchedRow) {
    // Return the formula based on location class
    if (locationClass === "I") return matchedRow.class_i;
    if (locationClass === "II") return matchedRow.class_ii;
    if (locationClass === "III") return matchedRow.class_iii;
    if (locationClass === "IV") return matchedRow.class_iv;
  }
  
  return "No matching rule";
}

// Lookup function for Table A.2 (Flammability)
function lookupTableA2(flammability, category, application, occupancyDesc, locationClass) {
  // Normalize flammability (handle both string "2" and number 2, "2L" etc.)
  const flamVal = flammability === null || flammability === undefined || flammability === "" ? "" : String(flammability);
  
  if (!flamVal || !category || !locationClass) {
    return "No matching rule";
  }
  
  for (const row of tableA2) {
    const rowFlam = String(row.flammability);
    const matchFlam = rowFlam === flamVal;
    const matchCat = row.category === category;
    
    // Match application - null in table means any application
    // If user didn't provide application, treat as null
    const userApp = application === "" || application === null ? null : application;
    const matchApp = row.application === null || 
                     (row.application !== null && userApp !== null && 
                      row.application.toLowerCase() === userApp.toLowerCase());
    
    // Match occupancy description - null means any
    const userOcc = occupancyDesc === "" || occupancyDesc === null ? null : occupancyDesc;
    const matchOcc = row.occupancy_desc === null || 
                     (row.occupancy_desc !== null && userOcc !== null && 
                      row.occupancy_desc === userOcc);
    
    if (matchFlam && matchCat && matchApp && matchOcc) {
      // Return the formula based on location class
      if (locationClass === "I") return row.class_i;
      if (locationClass === "II") return row.class_ii;
      if (locationClass === "III") return row.class_iii;
      if (locationClass === "IV") return row.class_iv;
    }
  }
  return "No matching rule";
}

// Calculate toxicity limit charge (Sheet0 logic)
// This mimics Sheet0!F2:G9 lookup table
// Sheet0 formulas:
// G2 = C4 * C5 = Toxicity Limit * Space Volume
// G4 = IF(B6="Sealed Absorption System", MIN(C4*C5, 2.5), G2)
function calculateToxicityCharge(formula, toxicityLimit, spaceVolume, systemType) {
  if (!formula || formula === "No matching rule" || formula === "ND") {
    return null;
  }
  
  // Handle "Change Location Class..." - returns null/empty
  if (formula.includes("Change Location Class")) {
    return null;
  }
  
  // G3: "No charge restriction" -> "NCR"
  if (formula === "No charge restriction" || formula === "No charge restrictiona") {
    return "NCR";
  }
  
  // G2: "Toxicity limit x Location volume" -> C4 * C5 = Toxicity Limit * Space Volume
  if (formula === "Toxicity limit x Location volume") {
    if (!isNumeric(toxicityLimit) || !isNumeric(spaceVolume)) {
      return null;
    }
    return toxicityLimit * spaceVolume;
  }
  
  // G4: "Toxicity Limit x Location Volume and not more than 2.5 kg for Sealed absorption systems"
  // Formula: IF(B6="Sealed Absorption System", MIN(C4*C5, 2.5), G2)
  if (formula === "Toxicity Limit x Location Volume and not more than 2.5 kg for Sealed absorption systems") {
    if (!isNumeric(toxicityLimit) || !isNumeric(spaceVolume)) {
      return null;
    }
    const baseResult = toxicityLimit * spaceVolume;
    if (systemType === "Sealed Absorption System") {
      return Math.min(baseResult, 2.5);
    } else {
      // If not Sealed Absorption System, use G2 (same as "Toxicity limit x Location volume")
      return baseResult;
    }
  }
  
  // G5: "Not more than 10 kg" -> 10
  if (formula === "Not more than 10 kg") {
    return 10;
  }
  
  // G7: "Not more than 25 kg" -> 25
  if (formula === "Not more than 25 kg") {
    return 25;
  }
  
  // G6: "Not more than 50 kg and Emer. Exits available" -> 50
  if (formula === "Not more than 50 kg and Emer. Exits available") {
    return 50;
  }
  
  // Handle variations (case-insensitive matching for robustness)
  if (formula.toLowerCase().includes("not more than")) {
    const match = formula.match(/not more than ([\d.]+)\s*kg/i);
    if (match) {
      return parseFloat(match[1]);
    }
    // Fallback for common cases
    if (formula.toLowerCase().includes("10 kg")) return 10;
    if (formula.toLowerCase().includes("25 kg")) return 25;
    if (formula.toLowerCase().includes("50 kg")) return 50;
  }
  
  // Default: return null for unmatched formulas
  return null;
}

// Calculate flammability limit charge (Sheet1 logic)
function calculateFlammabilityCharge(formula, lfl, floorArea, spaceVolume, h0, m1, m2, m3, systemType, occupancyCategory) {
  if (!formula || formula === "No matching rule" || formula === "ND" || formula.includes("Change Location")) {
    return null;
  }
  
  if (formula === "No charge restriction") {
    return null; // Return null to indicate no restriction (will be handled in MIN calculation)
  }
  
  // Calculate A.4 formula: 2.5 * (LFL^(5/4)) * h0 * (A^(1/5))
  function calculateA4(lfl, h0, floorArea) {
    if (!isNumeric(lfl) || !isNumeric(h0) || !isNumeric(floorArea)) return null;
    return 2.5 * Math.pow(lfl, 5/4) * h0 * Math.pow(floorArea, 1/5);
  }
  
  // Calculate 20% x LFL x Room volume
  function calculate20Percent(lfl, spaceVolume) {
    if (!isNumeric(lfl) || !isNumeric(spaceVolume)) return null;
    return 0.2 * lfl * spaceVolume;
  }
  
  const a4Value = calculateA4(lfl, h0, floorArea);
  const percent20Value = calculate20Percent(lfl, spaceVolume);
  
  // Handle different formula patterns (order matters - check more specific first)
  if (formula.includes("According to A.4 and not more than m2 x 1,5")) {
    if (!isNumeric(m2)) return null;
    const limit = m2 * 1.5;
    if (a4Value === null) return null;
    return Math.min(a4Value, limit);
  }
  
  if (formula.includes("According to A.4 and not more than m2")) {
    if (!isNumeric(m2)) return null;
    const limit = m2;
    if (a4Value === null) return null;
    return Math.min(a4Value, limit);
  }
  
  if (formula.includes("20 % x LFL x Room volume and not more than m2 x 1,5") || 
      formula === "20 % x LFL x Room volume and not more than m2 x 1,5 ") {
    if (!isNumeric(m2)) return null;
    const limit = m2 * 1.5;
    if (percent20Value === null) return null;
    return Math.min(percent20Value, limit);
  }
  
  if (formula.includes("20 % x LFL x Room volume and not more than m2") && 
      !formula.includes("25 kg") && !formula.includes("10 kg") && !formula.includes("50 kg")) {
    if (!isNumeric(m2)) return null;
    const limit = m2;
    if (percent20Value === null) return null;
    return Math.min(percent20Value, limit);
  }
  
  if (formula.includes("20 % x LFL x Room volume and not more than 25 kg")) {
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 25);
  }
  
  if (formula.includes("20 % x LFL x Room volume and not more than 10 kg")) {
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 10);
  }
  
  if (formula.includes("20 % x LFL x Room volume and not more than 50 kg")) {
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 50);
  }
  
  if (formula.includes("20% x LFL x Room volume and not more than")) {
    if (percent20Value === null) return null;
    const match = formula.match(/not more than ([\d.]+)\s*kg/);
    if (match) {
      const limit = parseFloat(match[1]);
      return Math.min(percent20Value, limit);
    }
  }
  
  if (formula.includes("20% LFLx Room volume and not more than 1 kg")) {
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 1);
  }
  
  if (formula.includes("20% x LFLx Room volume and not more than 2,5 kg")) {
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 2.5);
  }
  
  if (formula === "not more than m3 x 1,5") {
    if (!isNumeric(m3)) return null;
    return m3 * 1.5;
  }
  
  if (formula === "not more than m3") {
    if (!isNumeric(m3)) return null;
    return m3;
  }
  
  if (formula.includes("Only sealed systems: 20% x LFL x Room volume and not more than 1 kg")) {
    if (systemType !== "Sealed Absorption System") return null;
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 1);
  }
  
  if (formula.includes("Only sealed systems: 20% x LFL x Room volume and not more than 1,5 kg")) {
    if (systemType !== "Sealed Absorption System") return null;
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 1.5);
  }
  
  if (formula.includes("20% x LFL x Room volume and not more than 1 kg") && 
      !formula.includes("Only sealed systems")) {
    if (percent20Value === null) return null;
    return Math.min(percent20Value, 1);
  }
  
  if (formula === "Not more than 1 kg") {
    return 1;
  }
  
  if (formula === "Not more than 5 kg") {
    return 5;
  }
  
  if (formula === "Not more than 10 kg") {
    return 10;
  }
  
  if (formula.includes("In accordance with occupancy category a, other applications")) {
    if (!isNumeric(m2)) return null;
    const limit = m2 * 1.5;
    if (a4Value === null) return null;
    return Math.min(a4Value, limit);
  }
  
  if (formula.includes("In accordance with occupancy category b, other applications")) {
    if (!isNumeric(m2)) return null;
    const limit = m2 * 1.5;
    if (a4Value === null) return null;
    return Math.min(a4Value, limit);
  }
  
  if (formula.includes("In accordance with occupancy category c, other applications")) {
    if (!isNumeric(m2)) return null;
    const limit = m2 * 1.5;
    if (a4Value === null) return null;
    return Math.min(a4Value, limit);
  }
  
  return null;
}

// Main calculation function
function calculate() {
  // Get input values
  const F4 = document.getElementById("F4").value; // Occupancy Category
  const F6 = document.getElementById("F6").value; // Application
  const F7 = document.getElementById("F7").value; // Occupancy Description
  const F8 = document.getElementById("F8").value; // Location Class
  const F10 = document.getElementById("F10").value; // Refrigerant No.
  const F17 = parseValue(document.getElementById("F17").value); // Floor Area
  const F18 = parseValue(document.getElementById("F18").value); // Ceiling Height
  const F19 = parseValue(document.getElementById("F19").value) || 1; // Net Volume Ratio
  const F21 = document.getElementById("F21").value; // System Type
  const F22 = document.getElementById("F22").value; // Mount Type
  const F24 = parseValue(document.getElementById("F24").value); // Actual Charge
  
  // F5: Category (from Occupancy Category lookup)
  const F5 = occupancyCategoryTable[F4] || "";
  document.getElementById("F5").value = F5;
  
  // F9: Class (from Location Class lookup)
  const F9 = locationClassTable[F8] || "";
  document.getElementById("F9").value = F9;
  
  // F11-F15: Refrigerant Parameters
  const refParams = refrigerantParamsTable[F10] || {};
  const F11 = refParams.toxicity || ""; // Toxicity
  const F12 = refParams.flammability !== undefined ? refParams.flammability : ""; // Flammability
  const F13 = refParams.practical_l !== undefined && isNumeric(refParams.practical_l) ? refParams.practical_l : ""; // Practical L
  const F14 = refParams.atel_odl !== undefined && isNumeric(refParams.atel_odl) ? refParams.atel_odl : ""; // ATEL/ODL
  const F15 = refParams.lfl !== undefined && isNumeric(refParams.lfl) ? refParams.lfl : ""; // LFL
  
  document.getElementById("F11").value = F11;
  document.getElementById("F12").value = F12;
  document.getElementById("F13").value = F13;
  document.getElementById("F14").value = F14;
  document.getElementById("F15").value = F15;
  
  // F16: Toxicity Limit = MAX(F13, F14)
  const F13_num = parseValue(F13);
  const F14_num = parseValue(F14);
  const F16 = (F13_num !== null || F14_num !== null) ? Math.max(F13_num || 0, F14_num || 0) : "";
  document.getElementById("F16").value = F16;
  
  // F20: Space Volume = F17 * F18 * F19
  const F20 = (F17 !== null && F18 !== null && F19 !== null) ? F17 * F18 * F19 : "";
  document.getElementById("F20").value = F20;
  
  // F23: h0 based on Mount Type
  let F23 = "";
  if (F22 === "Floor") F23 = 0.6;
  else if (F22 === "Window") F23 = 1;
  else if (F22 === "Wall") F23 = 1.8;
  else if (F22 === "Ceiling") F23 = 2.2;
  else if (F22) F23 = "Select Location";
  document.getElementById("F23").value = F23;
  
  // F25-F27: Cap Factors (m1, m2, m3)
  const F15_num = parseValue(F15);
  const F25 = (F15_num !== null && isNumeric(F15_num)) ? 4 * F15_num : "NF"; // m1 = 4 * F15
  const F26 = (F15_num !== null && isNumeric(F15_num)) ? 26 * F15_num : "NF"; // m2 = 26 * F15
  const F27 = (F15_num !== null && isNumeric(F15_num)) ? 130 * F15_num : "NF"; // m3 = 130 * F15
  document.getElementById("F25").value = F25;
  document.getElementById("F26").value = F26;
  document.getElementById("F27").value = F27;
  
  // F28: Maximum Charge Formula from Table A.1
  const F28 = lookupTableA1(F11, F5, F7, F9);
  document.getElementById("F28").value = F28 || "";
  
  // F29: Max. Charge (Toxicity) - Calculate based on F28 formula
  // Ensure F16 and F20 are numbers for calculation
  const F16_num = typeof F16 === "number" ? F16 : (F16 === "" ? null : parseValue(F16));
  const F20_num = typeof F20 === "number" ? F20 : (F20 === "" ? null : parseValue(F20));
  const F29 = calculateToxicityCharge(F28, F16_num, F20_num, F21);
  document.getElementById("F29").value = F29 !== null && F29 !== undefined ? (typeof F29 === "string" ? F29 : F29.toFixed(10).replace(/\.?0+$/, "")) : "";
  
  // F30: Maximum Charge Formula from Table A.2
  const F30 = lookupTableA2(F12, F5, F6, F7, F9);
  document.getElementById("F30").value = F30 || "";
  
  // F31: Max. Charge (Flammability)
  let F31 = null;
  
  // Special case: IF(AND(F4="Machinery Room",OR(F12="2L",F12=2,F12=3)),0.2*F15*F20,...)
  if (F4 === "Machinery Room" && (F12 === "2L" || F12 === 2 || F12 === "2" || F12 === 3 || F12 === "3")) {
    if (isNumeric(F15_num) && isNumeric(F20)) {
      F31 = 0.2 * F15_num * F20;
    }
  } else {
    // Normal lookup
    const m1_num = parseValue(F25);
    const m2_num = parseValue(F26);
    const m3_num = parseValue(F27);
    const h0_num = parseValue(F23);
    
    F31 = calculateFlammabilityCharge(F30, F15_num, F17, F20, h0_num, m1_num, m2_num, m3_num, F21, F5);
  }
  
  document.getElementById("F31").value = F31 !== null ? (typeof F31 === "string" ? F31 : F31.toFixed(10).replace(/\.?0+$/, "")) : "NF";
  
  // F33: Refrigerant Charge Limit = MIN(F29, F31)
  // Handle "NCR" (No Charge Restriction) - if either is NCR, the other value is used
  // If both are NCR or null, return empty
  const F29_num = (typeof F29 === "string" && F29 === "NCR") ? null : (typeof F29 === "string" ? null : parseValue(F29));
  const F31_num = (typeof F31 === "string" && F31 === "NF") ? null : (typeof F31 === "string" ? null : parseValue(F31));
  
  let F33 = "";
  if (F29_num !== null && F31_num !== null) {
    F33 = Math.min(F29_num, F31_num);
  } else if (F29_num !== null && F29 !== "NCR") {
    F33 = F29_num;
  } else if (F31_num !== null && F31 !== "NF") {
    F33 = F31_num;
  } else if (F29 === "NCR" && F31_num !== null) {
    F33 = F31_num; // NCR means no restriction, so use the other value
  } else if (F31 === "NF" && F29_num !== null) {
    F33 = F29_num; // NF means not applicable, so use the other value
  } else if (F29 === "NCR" || (F31 === "NF" && F29_num === null)) {
    F33 = "NCR"; // Both are unrestricted or invalid
  }
  
  document.getElementById("F33").value = F33 !== "" ? (typeof F33 === "number" ? F33.toFixed(10).replace(/\.?0+$/, "") : F33) : "";
  
  // F34: Concentration Limit = F33 / F20
  let F34 = "";
  if (F33 !== "" && typeof F33 !== "string" && isNumeric(F33) && isNumeric(F20) && F20 !== 0) {
    F34 = parseFloat(F33) / F20;
    document.getElementById("F34").value = F34.toFixed(10).replace(/\.?0+$/, "");
  } else {
    document.getElementById("F34").value = "";
  }
}

// Initialize on page load
document.addEventListener("DOMContentLoaded", () => {
  // Populate dropdowns
  const F4_select = document.getElementById("F4");
  Object.keys(occupancyCategoryTable).forEach(key => {
    const option = document.createElement("option");
    option.value = key;
    option.textContent = key;
    F4_select.appendChild(option);
  });
  
  const F8_select = document.getElementById("F8");
  Object.keys(locationClassTable).forEach(key => {
    const option = document.createElement("option");
    option.value = key;
    option.textContent = key;
    F8_select.appendChild(option);
  });
  
  const F10_select = document.getElementById("F10");
  Object.keys(refrigerantParamsTable).forEach(key => {
    const option = document.createElement("option");
    option.value = key;
    option.textContent = key;
    F10_select.appendChild(option);
  });
  
  // Set default values (matching Excel)
  document.getElementById("F4").value = "Residential";
  document.getElementById("F6").value = "Human Comfort";
  document.getElementById("F7").value = "Above Ground";
  document.getElementById("F8").value = "Mechanical Equipment within the occupied Space";
  document.getElementById("F10").value = "R1234ze(E)";
  document.getElementById("F17").value = "15";
  document.getElementById("F18").value = "2.5";
  document.getElementById("F19").value = "1";
  document.getElementById("F21").value = "Other System";
  document.getElementById("F22").value = "Ceiling";
  document.getElementById("F24").value = "50";
  
  // Add event listeners
  document.querySelectorAll("input, select").forEach(el => {
    el.addEventListener("input", calculate);
    el.addEventListener("change", calculate);
  });
  
  // Run initial calculation
  calculate();
});
